<?php
declare(strict_types=1);

require_once __DIR__ . '/../app/core/bootstrap.php';
$user = require_role(['admin']);

header('Content-Type: application/json; charset=utf-8');

function fetch_one(string $sql, array $params=[]): float {
  $st = db()->prepare($sql);
  $st->execute($params);
  return (float)($st->fetchColumn() ?: 0);
}
function fetch_all(string $sql, array $params=[]): array {
  $st = db()->prepare($sql);
  $st->execute($params);
  return $st->fetchAll(PDO::FETCH_ASSOC) ?: [];
}

// ===== KPIs base =====
$total_cases   = (int)fetch_one("SELECT COUNT(*) FROM cases");
$total_capital = fetch_one("SELECT IFNULL(SUM(amount),0) FROM cases");
$total_paid    = fetch_one("SELECT IFNULL(SUM(amount),0) FROM case_payments");
$paid_30       = fetch_one("SELECT IFNULL(SUM(amount),0) FROM case_payments WHERE payment_date >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)");

$open_cases    = (int)fetch_one("SELECT COUNT(*) FROM cases WHERE status <> 'cerrado'");
$closed_cases  = (int)fetch_one("SELECT COUNT(*) FROM cases WHERE status = 'cerrado'");
$mora_cases    = (int)fetch_one("SELECT COUNT(*) FROM cases WHERE status = 'mora'");
$parcial_cases = (int)fetch_one("SELECT COUNT(*) FROM cases WHERE status = 'parcial'");

// ===== Serie cobrado últimos 14 días =====
$rows = fetch_all("
  SELECT payment_date d, SUM(amount) s
  FROM case_payments
  WHERE payment_date >= DATE_SUB(CURDATE(), INTERVAL 14 DAY)
  GROUP BY payment_date
  ORDER BY payment_date ASC
");

$labels = [];
$series_paid = [];
$map = [];
foreach ($rows as $r) $map[(string)$r['d']] = (float)$r['s'];

for ($i = 13; $i >= 0; $i--) {
  $d = date('Y-m-d', strtotime("-{$i} day"));
  $labels[] = date('d/m', strtotime($d));
  $series_paid[] = (float)($map[$d] ?? 0);
}

// ===== Donut por estado =====
$stateRows = fetch_all("
  SELECT status, COUNT(*) c
  FROM cases
  GROUP BY status
");
$state_labels = [];
$state_values = [];
foreach ($stateRows as $r) {
  $state_labels[] = (string)$r['status'];
  $state_values[] = (int)$r['c'];
}

// ===== Top clientes por saldo base (capital - pagos) =====
$topClients = fetch_all("
  SELECT c.id, c.name,
         (IFNULL(SUM(cs.amount),0) - IFNULL(SUM(p.sum_paid),0)) AS pending
  FROM clients c
  LEFT JOIN cases cs ON cs.client_id = c.id
  LEFT JOIN (
    SELECT case_id, SUM(amount) sum_paid
    FROM case_payments
    GROUP BY case_id
  ) p ON p.case_id = cs.id
  GROUP BY c.id
  HAVING pending > 0
  ORDER BY pending DESC
  LIMIT 8
");
$client_labels = [];
$client_values = [];
foreach ($topClients as $r) {
  $client_labels[] = (string)$r['name'];
  $client_values[] = (float)$r['pending'];
}

// ===== Top gestores por cobrado últimos 30 días (casos asignados activos) =====
$topGestors = fetch_all("
  SELECT u.id, u.name,
         IFNULL(SUM(p.amount),0) AS paid_30
  FROM users u
  LEFT JOIN case_assignments a
    ON a.gestor_user_id = u.id AND a.unassigned_at IS NULL
  LEFT JOIN case_payments p
    ON p.case_id = a.case_id AND p.payment_date >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)
  WHERE u.role='gestor'
  GROUP BY u.id
  ORDER BY paid_30 DESC
  LIMIT 8
");
$gestor_labels = [];
$gestor_values = [];
foreach ($topGestors as $r) {
  $gestor_labels[] = (string)$r['name'];
  $gestor_values[] = (float)$r['paid_30'];
}

// ===== Respuesta =====
echo json_encode([
  'kpis' => [
    'total_cases' => $total_cases,
    'open_cases' => $open_cases,
    'closed_cases' => $closed_cases,
    'mora_cases' => $mora_cases,
    'parcial_cases' => $parcial_cases,
    'total_capital' => $total_capital,
    'total_paid' => $total_paid,
    'paid_30' => $paid_30,
    'pending_base' => max(0, $total_capital - $total_paid),
  ],
  'charts' => [
    'paid14' => ['labels' => $labels, 'values' => $series_paid],
    'states' => ['labels' => $state_labels, 'values' => $state_values],
    'top_clients_pending' => ['labels' => $client_labels, 'values' => $client_values],
    'top_gestors_paid30' => ['labels' => $gestor_labels, 'values' => $gestor_values],
  ],
], JSON_UNESCAPED_UNICODE);
