<?php
declare(strict_types=1);

require_once __DIR__ . '/../app/core/bootstrap.php';
require_once __DIR__ . '/../app/models/ImportModel.php';
require_once __DIR__ . '/../app/models/ChatModel.php'; // opcional, por si quieres avisos en chat luego

$user = require_role(['admin']);
if ($_SERVER['REQUEST_METHOD'] !== 'POST') { http_response_code(405); exit('Method not allowed'); }
csrf_verify();

$id = (int)($_POST['id'] ?? 0);
$job = ImportModel::find($id);
if (!$job) { http_response_code(404); exit('Import no existe'); }

$path = __DIR__ . '/../' . ltrim((string)$job['stored_path'], '/');
$real = realpath($path);
if (!$real || !is_file($real)) { exit('CSV no encontrado'); }

// mapping
$mapping = [
  'client_name'     => (string)($_POST['map_client_name'] ?? ''),
  'cedula'          => (string)($_POST['map_cedula'] ?? ''),
  'debtor_name'     => (string)($_POST['map_debtor_name'] ?? ''),
  'phone'           => (string)($_POST['map_phone'] ?? ''),
  'address'         => (string)($_POST['map_address'] ?? ''),
  'amount'          => (string)($_POST['map_amount'] ?? ''),
  'debt_start_date' => (string)($_POST['map_debt_start_date'] ?? ''),
  'status'          => (string)($_POST['map_status'] ?? ''),
];

// opciones
$options = [
  'fixed_client_id'  => (int)($_POST['fixed_client_id'] ?? 0),
  'assign_mode'      => (string)($_POST['assign_mode'] ?? 'none'), // none|fixed|round_robin
  'fixed_gestor_id'  => (int)($_POST['fixed_gestor_id'] ?? 0),
  'rr_gestors'       => array_map('intval', (array)($_POST['rr_gestors'] ?? [])),
  'create_debtor_user' => (int)($_POST['create_debtor_user'] ?? 0) === 1,
  'default_password' => 'ppc123',
];

ImportModel::setMapped($id, $mapping, $options);
ImportModel::setRunning($id);

$delim = csv_detect_delimiter($real);
$fh = fopen($real, 'rb');
if (!$fh) { ImportModel::setError($id, 'No se pudo abrir CSV'); exit('No se pudo abrir CSV'); }

$header = fgetcsv($fh, 0, $delim);
if (!$header) { fclose($fh); ImportModel::setError($id, 'CSV sin encabezado'); exit('CSV sin encabezado'); }
if (isset($header[0])) $header[0] = preg_replace('/^\xEF\xBB\xBF/', '', (string)$header[0]);

$idx = [];
foreach ($header as $i => $h) $idx[(string)$h] = $i;

function col(array $row, array $idx, string $colName): string {
  if ($colName === '' || !isset($idx[$colName])) return '';
  $i = (int)$idx[$colName];
  return isset($row[$i]) ? trim((string)$row[$i]) : '';
}

function get_or_create_client(string $name): int {
  $name = trim($name);
  if ($name === '') return 0;
  $st = db()->prepare("SELECT id FROM clients WHERE name=? LIMIT 1");
  $st->execute([$name]);
  $id = (int)($st->fetchColumn() ?: 0);
  if ($id) return $id;

  $st2 = db()->prepare("INSERT INTO clients (name) VALUES (?)");
  $st2->execute([$name]);
  return (int)db()->lastInsertId();
}

// crea o obtiene user deudor por cédula (si tu esquema lo soporta)
function ensure_debtor_user(string $cedula, string $name, string $password): int {
  $cedula = trim($cedula);
  if ($cedula === '') return 0;

  // intenta por cedula (si existe columna)
  $cols = db()->query("SHOW COLUMNS FROM users")->fetchAll() ?: [];
  $hasCed = false;
  foreach ($cols as $c) { if ((string)$c['Field'] === 'cedula') { $hasCed = true; break; } }

  if (!$hasCed) return 0;

  $st = db()->prepare("SELECT id FROM users WHERE role='deudor' AND cedula=? LIMIT 1");
  $st->execute([$cedula]);
  $id = (int)($st->fetchColumn() ?: 0);
  if ($id) return $id;

  $hash = password_hash($password, PASSWORD_DEFAULT);
  $st2 = db()->prepare("INSERT INTO users (role, name, email, cedula, password_hash) VALUES ('deudor', ?, NULL, ?, ?)");
  $st2->execute([$name ?: $cedula, $cedula, $hash]);
  return (int)db()->lastInsertId();
}

$total = 0; $inserted = 0; $errors = 0;
$log = [];

$rr = $options['rr_gestors'];
$rrPos = 0;

db()->beginTransaction();
try {
  while (!feof($fh)) {
    $row = fgetcsv($fh, 0, $delim);
    if (!$row) continue;
    $total++;

    $cedula = col($row, $idx, $mapping['cedula']);
    $debtor = col($row, $idx, $mapping['debtor_name']);
    $phone  = col($row, $idx, $mapping['phone']);
    $addr   = col($row, $idx, $mapping['address']);
    $amtStr = col($row, $idx, $mapping['amount']);
    $amount = norm_money($amtStr);

    if ($cedula === '' || $amount <= 0) {
      $errors++;
      $log[] = "Fila {$total}: inválida (cedula/amount)";
      continue;
    }

    // cliente
    $client_id = (int)$options['fixed_client_id'];
    if ($client_id <= 0) {
      $clientName = col($row, $idx, $mapping['client_name']);
      $client_id = get_or_create_client($clientName);
    }
    if ($client_id <= 0) {
      $errors++;
      $log[] = "Fila {$total}: sin cliente";
      continue;
    }

    // estado / fecha inicio
    $status = col($row, $idx, $mapping['status']);
    if ($status === '') $status = 'mora';

    $debtStart = col($row, $idx, $mapping['debt_start_date']);
    $debtStart = $debtStart !== '' ? $debtStart : null;

    // crear user deudor (opcional)
    $debtor_user_id = null;
    if ($options['create_debtor_user']) {
      $uid = ensure_debtor_user($cedula, $debtor, $options['default_password']);
      if ($uid > 0) $debtor_user_id = $uid;
    }

    // insertar caso (ajusta nombres de columnas según tu schema real)
    $st = db()->prepare("
      INSERT INTO cases
      (client_id, debtor_user_id, debtor_name, cedula, phone, address, amount, debt_start_date, status, created_at)
      VALUES
      (?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())
    ");
    $st->execute([
      $client_id,
      $debtor_user_id,
      $debtor ?: $cedula,
      $cedula,
      $phone ?: null,
      $addr ?: null,
      $amount,
      $debtStart,
      $status
    ]);

    $case_id = (int)db()->lastInsertId();
    $inserted++;

    // asignación gestor
    $assign_mode = (string)$options['assign_mode'];
    $gestor_id = 0;

    if ($assign_mode === 'fixed') {
      $gestor_id = (int)$options['fixed_gestor_id'];
    } elseif ($assign_mode === 'round_robin' && count($rr) > 0) {
      $gestor_id = (int)$rr[$rrPos % count($rr)];
      $rrPos++;
    }

    if ($gestor_id > 0) {
      // inserta asignación activa
      db()->prepare("
        INSERT INTO case_assignments (case_id, gestor_user_id, assigned_by, assigned_at, unassigned_at)
        VALUES (?, ?, ?, NOW(), NULL)
      ")->execute([$case_id, $gestor_id, (int)$user['id']]);
    }
  }

  fclose($fh);
  db()->commit();

  ImportModel::setDone($id, $total, $inserted, $errors, implode("\n", array_slice($log, 0, 300)));
  redirect_to('admin/imports.php');

} catch (Throwable $e) {
  fclose($fh);
  db()->rollBack();
  ImportModel::setError($id, 'Error: ' . $e->getMessage());
  exit('Error importando: ' . $e->getMessage());
}
