<?php
// app/core/auth.php
declare(strict_types=1);

require_once __DIR__ . '/db.php';

function auth_user(): ?array {
  return isset($_SESSION['user']) && is_array($_SESSION['user']) ? $_SESSION['user'] : null;
}

function require_login(): array {
  $u = auth_user();
  if (!$u) {
    header('Location: login.php');
    exit;
  }
  return $u;
}

function require_role(array $roles): array {
  $u = require_login();
  if (!in_array($u['role'], $roles, true)) {
    http_response_code(403);
    exit('No autorizado.');
  }
  return $u;
}

/**
 * Login:
 * - Admin/Gestor: email + password
 * - Deudor: cédula + password (SI o SI)
 * Nota: permitimos que el input acepte "correo o cédula".
 */
function login_attempt(string $identifier, string $password): ?array {
  $identifier = trim($identifier);
  $is_email = (strpos($identifier, '@') !== false);

  if ($is_email) {
    $st = db()->prepare("SELECT * FROM users WHERE email = ? AND is_active=1 LIMIT 1");
    $st->execute([$identifier]);
  } else {
    $st = db()->prepare("SELECT * FROM users WHERE cedula = ? AND role='deudor' AND is_active=1 LIMIT 1");
    $st->execute([$identifier]);
  }

  $u = $st->fetch();
  if (!$u) return null;

  if (!password_verify($password, (string)$u['password_hash'])) return null;

  $_SESSION['user'] = [
    'id' => (int)$u['id'],
    'role' => (string)$u['role'],
    'name' => (string)$u['name'],
    'email' => $u['email'] ? (string)$u['email'] : null,
    'cedula' => $u['cedula'] ? (string)$u['cedula'] : null,
  ];

  db()->prepare("UPDATE users SET last_login_at = NOW() WHERE id = ?")->execute([(int)$u['id']]);

  return $_SESSION['user'];
}

function logout(): void {
  $_SESSION = [];
  if (ini_get("session.use_cookies")) {
    $p = session_get_cookie_params();
    setcookie(session_name(), '', time()-42000, $p["path"], $p["domain"], $p["secure"], $p["httponly"]);
  }
  session_destroy();
}
