<?php
declare(strict_types=1);

// ===========================
// HTML / Redirect
// ===========================
function h(string $s): string {
  return htmlspecialchars($s, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
}

function base_path(): string {
  static $bp = null;
  if ($bp !== null) return $bp;

  $cfgPath = __DIR__ . '/../config/config.php';
  $cfg = file_exists($cfgPath) ? (require $cfgPath) : [];
  $bp = rtrim((string)($cfg['base_path'] ?? ''), '/');
  return $bp;
}

function redirect_to(string $path): void {
  $bp = base_path();
  $path = ltrim($path, '/');
  $url = $bp ? "{$bp}/{$path}" : "/{$path}";
  header("Location: {$url}");
  exit;
}

// ===========================
// Settings + interés demo
// ===========================
function get_setting(string $k, $default = null) {
  $st = db()->prepare("SELECT v FROM settings WHERE k=? LIMIT 1");
  $st->execute([$k]);
  $v = $st->fetchColumn();
  return ($v !== false) ? $v : $default;
}

function calc_interest(float $principal, string $start_date): float {
  $pct = (float)get_setting('interest_monthly_pct', 0);
  if ($pct <= 0) return 0.0;

  $t0 = strtotime($start_date);
  if (!$t0) return 0.0;

  $months = max(0, (int)floor((time() - $t0) / (30 * 86400)));
  return round($principal * ($pct / 100) * $months, 2);
}

// ===========================
// Uploads CHAT (multi)
// ===========================
function handle_chat_uploads(int $case_id, int $message_id, array $files): void {
  require_once __DIR__ . '/../models/ChatModel.php';

  $baseDir = __DIR__ . '/../../storage/uploads/chat/case_' . $case_id;
  if (!is_dir($baseDir)) @mkdir($baseDir, 0775, true);

  $names = $files['name'] ?? [];
  $tmps  = $files['tmp_name'] ?? [];
  $errs  = $files['error'] ?? [];
  $sizes = $files['size'] ?? [];
  $types = $files['type'] ?? [];

  for ($i = 0; $i < count($names); $i++) {
    $err = (int)($errs[$i] ?? UPLOAD_ERR_NO_FILE);
    if ($err !== UPLOAD_ERR_OK) continue;

    $tmp = (string)($tmps[$i] ?? '');
    if (!$tmp || !is_uploaded_file($tmp)) continue;

    $origName = (string)($names[$i] ?? 'file');
    $size = (int)($sizes[$i] ?? 0);

    $finfo = finfo_open(FILEINFO_MIME_TYPE);
    $mime = $finfo ? (string)finfo_file($finfo, $tmp) : (string)($types[$i] ?? 'application/octet-stream');
    if ($finfo) finfo_close($finfo);

    $allowed = ['image/jpeg','image/png','image/webp','application/pdf'];
    if (!in_array($mime, $allowed, true)) continue;

    $ext = 'bin';
    if ($mime === 'application/pdf') $ext = 'pdf';
    elseif ($mime === 'image/jpeg') $ext = 'jpg';
    elseif ($mime === 'image/png') $ext = 'png';
    elseif ($mime === 'image/webp') $ext = 'webp';

    $safeBase = preg_replace('/[^a-zA-Z0-9_\-\.]+/', '_', pathinfo($origName, PATHINFO_FILENAME));
    $safeBase = $safeBase ? substr($safeBase, 0, 60) : 'file';

    $newName = $safeBase . '_' . date('Ymd_His') . '_' . bin2hex(random_bytes(4)) . '.' . $ext;
    $dest = $baseDir . '/' . $newName;

    if (!move_uploaded_file($tmp, $dest)) continue;

    $relative = 'storage/uploads/chat/case_' . $case_id . '/' . $newName;
    ChatModel::addAttachment($message_id, $origName, $relative, $mime, $size);
  }
}

// ===========================
// Uploads Pagos (1 archivo)
// ===========================
function handle_payment_attachment(int $case_id, array $file): ?string {
  $err = (int)($file['error'] ?? UPLOAD_ERR_NO_FILE);
  if ($err !== UPLOAD_ERR_OK) return null;

  $tmp = (string)($file['tmp_name'] ?? '');
  if (!$tmp || !is_uploaded_file($tmp)) return null;

  $baseDir = __DIR__ . '/../../storage/uploads/payments/case_' . $case_id;
  if (!is_dir($baseDir)) @mkdir($baseDir, 0775, true);

  $origName = (string)($file['name'] ?? 'file');

  $finfo = finfo_open(FILEINFO_MIME_TYPE);
  $mime = $finfo ? (string)finfo_file($finfo, $tmp) : 'application/octet-stream';
  if ($finfo) finfo_close($finfo);

  $allowed = ['image/jpeg','image/png','image/webp','application/pdf'];
  if (!in_array($mime, $allowed, true)) return null;

  $ext = 'bin';
  if ($mime === 'application/pdf') $ext = 'pdf';
  elseif ($mime === 'image/jpeg') $ext = 'jpg';
  elseif ($mime === 'image/png') $ext = 'png';
  elseif ($mime === 'image/webp') $ext = 'webp';

  $safeBase = preg_replace('/[^a-zA-Z0-9_\-\.]+/', '_', pathinfo($origName, PATHINFO_FILENAME));
  $safeBase = $safeBase ? substr($safeBase, 0, 60) : 'file';

  $newName = $safeBase . '_' . date('Ymd_His') . '_' . bin2hex(random_bytes(4)) . '.' . $ext;
  $dest = $baseDir . '/' . $newName;

  if (!move_uploaded_file($tmp, $dest)) return null;

  return 'storage/uploads/payments/case_' . $case_id . '/' . $newName;
}

// ===========================
// Recalcular status por saldo
// ===========================
function case_recalc_status(int $case_id): void {
  $st = db()->prepare("SELECT amount, created_at, debt_start_date FROM cases WHERE id=? LIMIT 1");
  $st->execute([$case_id]);
  $c = $st->fetch(PDO::FETCH_ASSOC);
  if (!$c) return;

  $amount = (float)$c['amount'];
  $start = (string)($c['debt_start_date'] ?: $c['created_at']);

  $st2 = db()->prepare("SELECT IFNULL(SUM(amount),0) FROM case_payments WHERE case_id=?");
  $st2->execute([$case_id]);
  $paid = (float)$st2->fetchColumn();

  $interest = calc_interest($amount, $start);
  $saldo = max(0, $amount + $interest - $paid);

  $status = 'mora';
  if ($saldo <= 0.00001) $status = 'cerrado';
  elseif ($paid > 0) $status = 'parcial';

  $st3 = db()->prepare("UPDATE cases SET status=? WHERE id=?");
  $st3->execute([$status, $case_id]);
}

// ===========================
// CSV helpers
// ===========================
function csv_detect_delimiter(string $path): string {
  $line = '';
  $fh = @fopen($path, 'rb');
  if ($fh) { $line = (string)fgets($fh); fclose($fh); }

  $candidates = [',',';','|',"\t"];
  $best = ','; $bestCount = -1;
  foreach ($candidates as $d) {
    $count = substr_count($line, $d);
    if ($count > $bestCount) { $bestCount = $count; $best = $d; }
  }
  return $best;
}

function csv_read_preview(string $path, int $maxRows = 10): array {
  $delim = csv_detect_delimiter($path);
  $fh = fopen($path, 'rb');
  if (!$fh) return [[], []];

  $header = fgetcsv($fh, 0, $delim);
  if (!$header) { fclose($fh); return [[], []]; }

  $rows = [];
  for ($i=0; $i < $maxRows && !feof($fh); $i++) {
    $r = fgetcsv($fh, 0, $delim);
    if (!$r) { $i--; continue; }
    $rows[] = $r;
  }
  fclose($fh);

  if (isset($header[0])) $header[0] = preg_replace('/^\xEF\xBB\xBF/', '', (string)$header[0]);
  return [$header, $rows];
}

function norm_money(string $s): float {
  $s = trim($s);
  $s = str_replace([' ', '₡', '$'], '', $s);

  if (substr_count($s, ',') > 0 && substr_count($s, '.') > 0) {
    $lastComma = strrpos($s, ',');
    $lastDot   = strrpos($s, '.');
    if ($lastComma > $lastDot) { $s = str_replace('.', '', $s); $s = str_replace(',', '.', $s); }
    else { $s = str_replace(',', '', $s); }
  } elseif (substr_count($s, ',') > 0 && substr_count($s, '.') === 0) {
    $s = str_replace('.', '', $s);
    $s = str_replace(',', '.', $s);
  } else {
    $s = str_replace(',', '', $s);
  }

  return (float)$s;
}
