<?php
// app/models/CaseModel.php
declare(strict_types=1);

final class CaseModel {

  public static function find(int $id): ?array {
    $st = db()->prepare("SELECT * FROM cases WHERE id=? LIMIT 1");
    $st->execute([$id]);
    $r = $st->fetch(PDO::FETCH_ASSOC);
    return $r ?: null;
  }

  public static function listAdmin(array $f = []): array {
    $q = trim((string)($f['q'] ?? ''));
    $status = trim((string)($f['status'] ?? ''));
    $client_id = (int)($f['client_id'] ?? 0);

    $where = [];
    $params = [];

    if ($q !== '') {
      $where[] = "(c.debtor_name LIKE ? OR c.cedula LIKE ? OR c.phone LIKE ? OR cl.name LIKE ?)";
      $params[] = "%{$q}%";
      $params[] = "%{$q}%";
      $params[] = "%{$q}%";
      $params[] = "%{$q}%";
    }
    if ($status !== '') {
      $where[] = "c.status = ?";
      $params[] = $status;
    }
    if ($client_id > 0) {
      $where[] = "c.client_id = ?";
      $params[] = $client_id;
    }

    $sql = "
      SELECT
        c.*,
        cl.name AS client_name,
        (SELECT u.name
         FROM case_assignments a
         JOIN users u ON u.id=a.gestor_user_id
         WHERE a.case_id=c.id AND a.unassigned_at IS NULL
         ORDER BY a.assigned_at DESC LIMIT 1
        ) AS gestor_name
      FROM cases c
      JOIN clients cl ON cl.id=c.client_id
    ";
    if ($where) $sql .= " WHERE " . implode(" AND ", $where);
    $sql .= " ORDER BY c.id DESC LIMIT 500";

    $st = db()->prepare($sql);
    $st->execute($params);
    return $st->fetchAll(PDO::FETCH_ASSOC) ?: [];
  }

  public static function listGestor(int $gestor_id, array $f = []): array {
    $q = trim((string)($f['q'] ?? ''));
    $status = trim((string)($f['status'] ?? ''));

    $where = [
      "a.gestor_user_id = ?",
      "a.unassigned_at IS NULL"
    ];
    $params = [$gestor_id];

    if ($q !== '') {
      $where[] = "(c.debtor_name LIKE ? OR c.cedula LIKE ? OR c.phone LIKE ? OR cl.name LIKE ?)";
      $params[] = "%{$q}%";
      $params[] = "%{$q}%";
      $params[] = "%{$q}%";
      $params[] = "%{$q}%";
    }
    if ($status !== '') {
      $where[] = "c.status = ?";
      $params[] = $status;
    }

    $sql = "
      SELECT c.*, cl.name AS client_name
      FROM case_assignments a
      JOIN cases c ON c.id=a.case_id
      JOIN clients cl ON cl.id=c.client_id
      WHERE " . implode(" AND ", $where) . "
      ORDER BY c.id DESC
      LIMIT 500
    ";
    $st = db()->prepare($sql);
    $st->execute($params);
    return $st->fetchAll(PDO::FETCH_ASSOC) ?: [];
  }

  public static function upsert(array $data, ?int $id = null): int {
    // Campos permitidos
    $fields = [
      'client_id','debtor_user_id','debtor_name','cedula','phone','address',
      'amount','debt_start_date','status'
    ];

    $row = [];
    foreach ($fields as $k) {
      if (array_key_exists($k, $data)) $row[$k] = $data[$k];
    }

    if ($id) {
      $sets = [];
      $params = [];
      foreach ($row as $k=>$v) { $sets[] = "{$k}=?"; $params[] = $v; }
      $params[] = $id;
      $sql = "UPDATE cases SET " . implode(',', $sets) . " WHERE id=?";
      db()->prepare($sql)->execute($params);
      return $id;
    }

    $cols = array_keys($row);
    $qs = array_fill(0, count($cols), '?');
    $params = array_values($row);

    $sql = "INSERT INTO cases (" . implode(',', $cols) . ") VALUES (" . implode(',', $qs) . ")";
    db()->prepare($sql)->execute($params);
    return (int)db()->lastInsertId();
  }

  public static function totals(int $case_id): array {
    $c = self::find($case_id);
    if (!$c) return ['paid'=>0.0,'interest'=>0.0,'saldo'=>0.0];

    $amount = (float)$c['amount'];
    $start = (string)($c['debt_start_date'] ?: $c['created_at']);

    $st = db()->prepare("SELECT IFNULL(SUM(amount),0) FROM case_payments WHERE case_id=?");
    $st->execute([$case_id]);
    $paid = (float)$st->fetchColumn();

    $interest = function_exists('calc_interest') ? (float)calc_interest($amount, $start) : 0.0;
    $saldo = max(0.0, $amount + $interest - $paid);

    return ['paid'=>$paid,'interest'=>$interest,'saldo'=>$saldo];
  }
}
