const input = document.getElementById('imagesInput');
const preview = document.getElementById('previewArea');
const btnPdf = document.getElementById('generatePdf');
const form = document.getElementById('expForm');

let images = [];

input.addEventListener('change', e => {
  preview.innerHTML = '';
  images = [];

  [...e.target.files].forEach(file => {
    const reader = new FileReader();
    reader.onload = ev => {
      const img = document.createElement('img');
      img.src = ev.target.result;
      img.className = 'm-2 rounded border';
      img.style.maxWidth = '180px';
      img.style.cursor = 'pointer';

      preview.appendChild(img);

      img.addEventListener('click', () => cropImage(img));
      images.push(img);
    };
    reader.readAsDataURL(file);
  });
});

function cropImage(img) {
  const modal = document.createElement('div');
  modal.innerHTML = `
    <div class="modal fade show" style="display:block;background:#0008">
      <div class="modal-dialog modal-lg">
        <div class="modal-content">
          <div class="modal-header">
            <h5 class="modal-title">Recortar documento</h5>
            <button type="button" class="close">&times;</button>
          </div>
          <div class="modal-body">
            <img id="cropImg" src="${img.src}" style="max-width:100%">
          </div>
          <div class="modal-footer">
            <button type="button" class="btn btn-primary">Aceptar</button>
          </div>
        </div>
      </div>
    </div>`;
  document.body.appendChild(modal);

  const cropImg = modal.querySelector('#cropImg');
  const cropper = new Cropper(cropImg, { viewMode: 1 });

  modal.querySelector('.btn-primary').onclick = () => {
    const canvas = cropper.getCroppedCanvas();
    applyScanFilter(canvas);
    img.src = canvas.toDataURL('image/jpeg', 0.9);
    document.body.removeChild(modal);
  };
  modal.querySelector('.close').onclick = () => document.body.removeChild(modal);
}

function applyScanFilter(canvas) {
  const ctx = canvas.getContext('2d');
  const imgData = ctx.getImageData(0, 0, canvas.width, canvas.height);
  const d = imgData.data;

  for (let i = 0; i < d.length; i += 4) {
    const gray = d[i] * 0.3 + d[i+1] * 0.59 + d[i+2] * 0.11;
    const val = gray > 150 ? 255 : 0;
    d[i] = d[i+1] = d[i+2] = val;
  }
  ctx.putImageData(imgData, 0, 0);
}

btnPdf.addEventListener('click', async () => {
  if (!images.length) {
    alert('Selecciona al menos una imagen.');
    return;
  }

  const { jsPDF } = window.jspdf;
  const pdf = new jsPDF('p', 'mm', 'a4');

  images.forEach((img, idx) => {
    if (idx > 0) pdf.addPage();
    pdf.addImage(img.src, 'JPEG', 10, 10, 190, 270);
  });

  const blob = pdf.output('blob');

  const fd = new FormData(form);
  fd.append('pdf', blob, 'expediente.pdf');

  const res = await fetch(form.action, { method: 'POST', body: fd });
  if (res.ok) {
    // vuelve al caso (el backend ya redirige)
    window.location.href = res.url || window.location.href;
  } else {
    alert('Error guardando el expediente.');
  }
});
