<?php
declare(strict_types=1);

require_once __DIR__ . '/../app/core/bootstrap.php';
$user = require_role(['gestor']);

header('Content-Type: application/json; charset=utf-8');

$uid = (int)$user['id'];

function fetch_one(string $sql, array $params=[]): float {
  $st = db()->prepare($sql);
  $st->execute($params);
  return (float)($st->fetchColumn() ?: 0);
}
function fetch_all(string $sql, array $params=[]): array {
  $st = db()->prepare($sql);
  $st->execute($params);
  return $st->fetchAll(PDO::FETCH_ASSOC) ?: [];
}

// Casos asignados activos
$assigned_cases = (int)fetch_one("
  SELECT COUNT(*)
  FROM case_assignments a
  WHERE a.gestor_user_id=? AND a.unassigned_at IS NULL
", [$uid]);

$capital = fetch_one("
  SELECT IFNULL(SUM(c.amount),0)
  FROM cases c
  JOIN case_assignments a ON a.case_id=c.id
  WHERE a.gestor_user_id=? AND a.unassigned_at IS NULL
", [$uid]);

$paid = fetch_one("
  SELECT IFNULL(SUM(p.amount),0)
  FROM case_payments p
  JOIN case_assignments a ON a.case_id=p.case_id
  WHERE a.gestor_user_id=? AND a.unassigned_at IS NULL
", [$uid]);

$paid_30 = fetch_one("
  SELECT IFNULL(SUM(p.amount),0)
  FROM case_payments p
  JOIN case_assignments a ON a.case_id=p.case_id
  WHERE a.gestor_user_id=? AND a.unassigned_at IS NULL
    AND p.payment_date >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)
", [$uid]);

$pending_base = max(0, $capital - $paid);

// Estados (solo casos asignados)
$stateRows = fetch_all("
  SELECT c.status, COUNT(*) c
  FROM cases c
  JOIN case_assignments a ON a.case_id=c.id
  WHERE a.gestor_user_id=? AND a.unassigned_at IS NULL
  GROUP BY c.status
", [$uid]);
$state_labels = [];
$state_values = [];
foreach ($stateRows as $r) { $state_labels[] = (string)$r['status']; $state_values[] = (int)$r['c']; }

// Serie cobrado 14 días (solo casos asignados)
$rows = fetch_all("
  SELECT p.payment_date d, SUM(p.amount) s
  FROM case_payments p
  JOIN case_assignments a ON a.case_id=p.case_id
  WHERE a.gestor_user_id=? AND a.unassigned_at IS NULL
    AND p.payment_date >= DATE_SUB(CURDATE(), INTERVAL 14 DAY)
  GROUP BY p.payment_date
  ORDER BY p.payment_date ASC
", [$uid]);

$labels = [];
$series_paid = [];
$map = [];
foreach ($rows as $r) $map[(string)$r['d']] = (float)$r['s'];
for ($i = 13; $i >= 0; $i--) {
  $d = date('Y-m-d', strtotime("-{$i} day"));
  $labels[] = date('d/m', strtotime($d));
  $series_paid[] = (float)($map[$d] ?? 0);
}

// Top 8 casos por pendiente base (capital - pagos) para tabla rápida
$topCases = fetch_all("
  SELECT c.id, c.debtor_name, c.cedula,
         (c.amount - IFNULL(p.paid,0)) AS pending
  FROM cases c
  JOIN case_assignments a ON a.case_id=c.id AND a.unassigned_at IS NULL
  LEFT JOIN (
    SELECT case_id, SUM(amount) paid
    FROM case_payments
    GROUP BY case_id
  ) p ON p.case_id=c.id
  WHERE a.gestor_user_id=?
  ORDER BY pending DESC
  LIMIT 8
", [$uid]);

echo json_encode([
  'kpis' => [
    'assigned_cases' => $assigned_cases,
    'capital' => $capital,
    'paid' => $paid,
    'paid_30' => $paid_30,
    'pending_base' => $pending_base
  ],
  'charts' => [
    'paid14' => ['labels' => $labels, 'values' => $series_paid],
    'states' => ['labels' => $state_labels, 'values' => $state_values],
  ],
  'top_cases' => $topCases,
], JSON_UNESCAPED_UNICODE);
