<?php
// public/install.php
declare(strict_types=1);
// Source - https://stackoverflow.com/q
// Posted by Abs, modified by community. See post 'Timeline' for change history
// Retrieved 2026-01-26, License - CC BY-SA 4.0

error_reporting(E_ALL);
ini_set('display_errors', 1);

/**
 * Instalador inicial (1 sola vez)
 * - Crea usuarios demo: Admin/Gestor/Deudor
 * - Password seguro con password_hash()
 *
 * IMPORTANTE:
 * - Ejecuta UNA vez: https://cobranza.jportales.com/sistema/install.php
 * - Luego BORRA este archivo del servidor.
 */

require_once __DIR__ . '/../app/core/bootstrap.php';

$cfg = require __DIR__ . '/../app/config/config.php';

// Anti-accidental: solo permitir si no hay admin todavía (o forzar con ?force=1)
$force = ((string)($_GET['force'] ?? '') === '1');

function has_admin(): bool {
  try {
    $st = db()->query("SELECT COUNT(*) FROM users WHERE role='admin'");
    return ((int)$st->fetchColumn()) > 0;
  } catch (Throwable $e) {
    return false;
  }
}

if (has_admin() && !$force) {
  http_response_code(403);
  echo "<h3>Instalación bloqueada</h3>";
  echo "<p>Ya existe un admin. Si de verdad deseas re-crear usuarios demo, usa <code>?force=1</code>.</p>";
  exit;
}

$admin_email = 'cobranza@jportales.com';
$gestor_email = 'gestor@jportales.com';
$deudor_email = 'deudor@jportales.com'; // opcional, no se usa para login de deudor
$pass_plain  = 'ppc123';

// Deudor login por cédula sí o sí:
$deudor_cedula = '1-1111-1111'; // puedes cambiarlo luego cuando importes el caso real

$pass_hash = password_hash($pass_plain, PASSWORD_DEFAULT);
if (!$pass_hash) {
  http_response_code(500);
  exit("No se pudo generar password_hash.");
}

db()->beginTransaction();

try {
  // Helper upsert simple (por email o cedula)
  $upsert_user = function(string $role, string $name, ?string $email, ?string $cedula) use ($pass_hash): int {
    // Buscar existente
    if ($role === 'deudor') {
      $st = db()->prepare("SELECT id FROM users WHERE cedula = ? LIMIT 1");
      $st->execute([$cedula]);
    } else {
      $st = db()->prepare("SELECT id FROM users WHERE email = ? LIMIT 1");
      $st->execute([$email]);
    }
    $id = $st->fetchColumn();

    if ($id) {
      // update
      $st2 = db()->prepare("
        UPDATE users
           SET role=?, name=?, email=?, cedula=?, password_hash=?, is_active=1, updated_at=NOW()
         WHERE id=?
      ");
      $st2->execute([$role, $name, $email, $cedula, $pass_hash, (int)$id]);
      return (int)$id;
    }

    // insert
    $st3 = db()->prepare("
      INSERT INTO users (role, name, email, cedula, password_hash, is_active, created_at)
      VALUES (?, ?, ?, ?, ?, 1, NOW())
    ");
    $st3->execute([$role, $name, $email, $cedula, $pass_hash]);
    return (int)db()->lastInsertId();
  };

  $admin_id = $upsert_user('admin', 'Com Cobranzas Admin', $admin_email, null);
  $gestor_id = $upsert_user('gestor', 'Gestor Demo', $gestor_email, null);
  $deudor_id = $upsert_user('deudor', 'Deudor Demo', $deudor_email, $deudor_cedula);

  // Crear cliente/cartera/caso demo si no existe (opcional pero útil)
  // Cliente BCR
  db()->exec("INSERT INTO clients (name, notes) VALUES ('BCR','Cliente demo') ON DUPLICATE KEY UPDATE notes=VALUES(notes)");

  $client_id = (int)db()->query("SELECT id FROM clients WHERE name='BCR' LIMIT 1")->fetchColumn();

  db()->prepare("
    INSERT INTO portfolios (client_id, name, imported_at, notes)
    VALUES (?, 'BCR - Cartera Demo', NOW(), 'Cartera demo')
    ON DUPLICATE KEY UPDATE notes=VALUES(notes)
  ")->execute([$client_id]);

  $portfolio_id = (int)db()->query("
    SELECT id FROM portfolios WHERE client_id={$client_id} AND name='BCR - Cartera Demo' LIMIT 1
  ")->fetchColumn();

  // Caso demo (cedula única global)
  // Si ya existe, actualizamos para linkearlo con el deudor user
  $stCase = db()->prepare("SELECT id FROM cases WHERE cedula=? LIMIT 1");
  $stCase->execute([$deudor_cedula]);
  $case_id = (int)($stCase->fetchColumn() ?: 0);

  if ($case_id > 0) {
    db()->prepare("
      UPDATE cases
         SET client_id=?, portfolio_id=?, debtor_user_id=?, debtor_name='Deudor Demo', status='nuevo', updated_at=NOW()
       WHERE id=?
    ")->execute([$client_id, $portfolio_id, $deudor_id, $case_id]);
  } else {
    db()->prepare("
      INSERT INTO cases (client_id, portfolio_id, debtor_user_id, cedula, debtor_name, phone, email, address, status, created_at)
      VALUES (?, ?, ?, ?, 'Deudor Demo', '8888-8888', ?, 'San José', 'nuevo', NOW())
    ")->execute([$client_id, $portfolio_id, $deudor_id, $deudor_cedula, $deudor_email]);
    $case_id = (int)db()->lastInsertId();
  }

  // Deuda demo (si no existe, crear)
  db()->prepare("
    INSERT INTO case_debt (case_id, capital, interest_rate_monthly, late_fee_rate_daily, fees, origin_date, overdue_date, notes)
    VALUES (?, 250000.00, 0.0300, 0.0010, 15000.00, '2025-01-01', '2025-03-01', 'Deuda demo')
    ON DUPLICATE KEY UPDATE capital=VALUES(capital), interest_rate_monthly=VALUES(interest_rate_monthly),
      late_fee_rate_daily=VALUES(late_fee_rate_daily), fees=VALUES(fees)
  ")->execute([$case_id]);

  // Asignar caso demo al gestor
  // (cierra asignaciones previas y crea una activa)
  db()->prepare("UPDATE case_assignments SET unassigned_at=NOW() WHERE case_id=? AND unassigned_at IS NULL")->execute([$case_id]);
  db()->prepare("
    INSERT INTO case_assignments (case_id, gestor_user_id, assigned_by, assigned_at)
    VALUES (?, ?, ?, NOW())
  ")->execute([$case_id, $gestor_id, $admin_id]);

  db()->commit();

} catch (Throwable $e) {
  db()->rollBack();
  http_response_code(500);
  echo "<h3>Error instalando</h3>";
  echo "<pre>".htmlspecialchars($e->getMessage(), ENT_QUOTES, 'UTF-8')."</pre>";
  exit;
}

// Mostrar credenciales (solo para esta ejecución)
?>
<!doctype html>
<html lang="es">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>Instalación completada</title>
  <style>
    body{font-family:system-ui,Segoe UI,Roboto,Arial;margin:20px;background:#f6f7fb}
    .card{background:#fff;border-radius:14px;padding:16px;max-width:760px;box-shadow:0 10px 30px rgba(0,0,0,.08)}
    code{background:#f2f2f2;padding:2px 6px;border-radius:8px}
    .warn{color:#b91c1c;font-weight:700}
  </style>
</head>
<body>
  <div class="card">
    <h2>✅ Instalación completada</h2>
    <p class="warn">IMPORTANTE: Borra <code>install.php</code> ahora mismo.</p>

    <h3>Credenciales demo</h3>
    <ul>
      <li><b>Admin</b>: <code><?= htmlspecialchars($admin_email) ?></code> / <code><?= htmlspecialchars($pass_plain) ?></code></li>
      <li><b>Gestor</b>: <code><?= htmlspecialchars($gestor_email) ?></code> / <code><?= htmlspecialchars($pass_plain) ?></code></li>
      <li><b>Deudor (login por cédula)</b>: <code><?= htmlspecialchars($deudor_cedula) ?></code> / <code><?= htmlspecialchars($pass_plain) ?></code></li>
    </ul>

    <p>
      Ir al login: <a href="login.php">login.php</a>
    </p>

    <p style="opacity:.8">
      Nota: el email del deudor se guarda como dato, pero el acceso del deudor siempre es por cédula.
    </p>
  </div>
</body>
</html>
